/** @file www_assert.h
 * @brief This header contains the assertion macro.
 *
 * This header holds the definition of the assert-macro for
 * We Want War - game.
 *
 * This assert-function uses the alert-function from Allegro library.
 *
 * @author      Tomi Lamminsaari
 * @version     $Revision: 1.2 $
 */
 

#ifndef H_WWW_ASSERT_H
#define H_WWW_ASSERT_H

#include <allegro.h>
#include <string>
#include <fstream>
#include "WWW_Defines.h"


namespace WeWantWar {


#ifdef RELEASE_VERSION

/** @def WWW_ASSERT( e )
 * This is the assertion-macro this game uses. If there is an assertion
 * error, this macro opens an Allegro alert-window and shows, where that
 * failure happened. If <code>NDEBUG</code> label has been defined, this
 * macro does nothing.
 */
#define WWW_ASSERT( e )
#define LOG_MESSAGE( mess )


#else

/** 
 * This is the assertion-macro this game uses. If there is an assertion
 * error, this macro opens an Allegro alert-window and shows, where that
 * failure happened.
 */
#define WWW_ASSERT( e ) WWW_Assertion( static_cast<bool>(e), #e, __FILE__, __LINE__ )

/**
 * Writes the given C-string file "log.txt"
 */
#define LOG_MESSAGE( mess ) \
        { \
          std::ofstream logfout("log.txt", std::ios::app ); \
          if ( !logfout == false ) { \
            logfout << (const char*) mess << std::endl; \
          } \
          logfout.close(); \
        }


#endif  // RELEASE_VERSION
    
    
/** The assertion-function itself. If the given condition is not true, we
 * show an alert window. This function allows user to continue the game
 * normally or abort it.
 * @pre
 * The Allegro-library must be initialized and there must be a screen mode set
 * with <code>set_gfx_mode(...)</code> function.
 * @param     e                   The boolean value for the condition
 * @param     pCond               A string that holds the condition in plain text.
 * @param     pFile               The current file we're compiling
 * @param     lineNum             The current linenumber
 */
inline void WWW_Assertion( bool e, const char* pCond, const char* pFile, unsigned int lineNum )
{
#ifndef RELEASE_VERSION
  if ( e == false ) {
    char lnum[6];
    itoa( __LINE__, lnum, 5 );
    std::string fileLine = std::string( pFile ) + std::string(", line ") +
                           std::string( lnum );
                           
    int ret = alert("Assertion failed !", pCond, fileLine.c_str(), "Continue", "Abort", 0,0);
    if ( ret == 1 ) {
      return;
      
    } else {
      abort();
    }
  }
#endif
}


};  // end of namespace

#endif  // H_WWW_ASSERT_H

/**
 * Version history
 * ===============
 * $Log: www_assert.h,v $
 * Revision 1.2  2006/08/20 21:12:50  lamminsa
 * Defines moved to new WWW_Defines.h header.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:22+02  lamminsa
 * Initial revision
 *
 */
 
